// thanks to btdsys, 7900, and usr for answering questions, but especially btdsys
// for making peerlib, without which this wouldn't be possible.

#define MACHINE_NAME "Peer Note-Pool"
#define AUTHOR "jmmcd"
#ifdef _DEBUG
#define FULL_NAME AUTHOR " " MACHINE_NAME " alpha"
#else
#define FULL_NAME AUTHOR " " MACHINE_NAME
#endif
#define MACHINE_VERSION "1.0"

#define ABOUT_TEXT FULL_NAME " " MACHINE_VERSION \
	"\nby James McDermott" \
	"\njamesmichaelmcdermott@eircom.net" \
	"\nwww.skynet.ie/~jmmcd" \
"\njmmcd on buzzmachines.com and buzzmusic.wipe-records.org and #buzz"

#define PARA_VOLUME_MAX 0xFE
#define PARA_VOLUME_DEF 0x80
#define MAX_TRACKS 128

#include "../mdk/mdk.h"
#include "peerctrl.h"
#include <float.h>
#include <time.h>
#include <stdlib.h>
#include <string.h>
#include "resource.h"

typedef enum {c, c_sh, d, d_sh, e, f, f_sh, g, g_sh, a, a_sh, b, off, numNotes} pitch_type;

////////////////////////////////////////////////////////////////////////
// Track Parameters

CMachineParameter const paraNote =
{
	pt_note,			//Type
		"Note",			//Short name
		"Note",				//Long name
		NOTE_MIN,					//Min value
		NOTE_MAX,				//Max value
		NOTE_NO,				//No value
		0,			//Flags
		NOTE_NO					//Default value
};

CMachineParameter const paraVolume =
{
	pt_byte,			//Type
		"Volume",			//Short name
		"Volume",				//Long name
		0,					//Min value
		PARA_VOLUME_MAX,				//Max value
		PARA_VOLUME_MAX + 1,				//No value
		MPF_STATE,			//Flags
		PARA_VOLUME_DEF				//Default value
};

CMachineParameter const paraProb =
{
	pt_byte,			//Type
		"Prob",			//Short name
		"Probability a note will be played this tick",	//Long name
		0,					//Min value
		100,				//Max value
		101,				//No value
		0,			//Flags: 0 means don't appear as a slider
		100				//Default value
};

CMachineParameter const paraCentre = {
	pt_note,			//Type
		"Centre",			//Short name
		"Centre Note",				//Long name
		NOTE_MIN,					//Min value
		NOTE_MAX,				//Max value
		NOTE_NO,				//No value
		MPF_STATE,			//Flags
		65					//Default value - middle C
};

CMachineParameter const paraOctDev = {
	pt_byte,			//Type
		"OctDev",			//Short name
		"Octave Randomisation amount [up to the amount specified]", //Long name
		0,					//Min value
		30,				//Max value
		31,				//No value
		MPF_STATE,			//Flags
		0				//Default value
};

CMachineParameter const paraVolDev = {
	pt_byte,			//Type
		"VolDev",			//Short name
		"Volume Randomisation amount", //Long name
		0,					//Min value
		100,				//Max value
		101,				//No value
		MPF_STATE,			//Flags
		0				//Default value
};

CMachineParameter const paraDotProb = {
	pt_byte,			//Type
		"DotProb",			//Short name
		"Probability that a note will be played on empty tick",				//Long name
		0,					//Min value
		0xFE,				//Max value
		0xFF,				//No value
		MPF_STATE,			//Flags
		30				//Default value
};

CMachineParameter const paraTarget = {
	pt_byte,			//Type
		"Target",			//Short name
		"Target track (for track parameters)",				//Long name
		0,					//Min value
		MAX_TRACKS,				//Max value
		MAX_TRACKS + 1,				//No value
		MPF_STATE,			//Flags FIXME: does tick-on-edit do anything?
		0				//Default value
};

CMachineParameter const paraOn = {
	pt_switch,			//Type
		"On",			//Short name
		"On/Off switch",				//Long name
		SWITCH_OFF,					//Min value
		SWITCH_ON,				//Max value
		SWITCH_NO,				//No value
		MPF_TICK_ON_EDIT | MPF_STATE,			//Flags FIXME: does tick-on-edit do anything?
		SWITCH_OFF				//Default value
};

////////////////////////////////////////////////////////////////////////
// global parameters

// 13: 12 notes and note-off
CMachineParameter paraNoteProb[numNotes];

////////////////////////////////////////////////////////////////////////
// Param-related pointer arrays and classes

CMachineParameter const *pParameters[] = {
	&paraNoteProb[c],
		&paraNoteProb[c_sh],
		&paraNoteProb[d],
		&paraNoteProb[d_sh],
		&paraNoteProb[e],
		&paraNoteProb[f],
		&paraNoteProb[f_sh],
		&paraNoteProb[g],
		&paraNoteProb[g_sh],
		&paraNoteProb[a],
		&paraNoteProb[a_sh],
		&paraNoteProb[b],
		&paraNoteProb[off],
		&paraNote,
		&paraVolume,
		&paraProb,
		&paraCentre,
		&paraOctDev,
		&paraVolDev,
		&paraDotProb,
		&paraTarget,
		&paraOn,
		NULL
};

CMachineAttribute const *pAttributes[] = {
	NULL
};

// a neat trick to set index-values and counts for all the parameters and attributes at once.
enum {
	numNoteProb = 0,
		NUMG = numNoteProb + numNotes,
		numNote = NUMG,
		numVolume,
		numProb,
		numCentre,
		numOctDev,
		numVolDev,
		numDotProb,
		numTarget,
		numOn,
		NUMT = numOn - NUMG + 1, // wouldn't it be easier to just count?
		NUMA = 0
};

#pragma pack(1)

class gvals
{
public:
	byte note_prob[numNotes];
};

class tvals
{
public:
	byte note;
	byte volume;
	byte prob;
	byte centre_note;
	byte oct_dev;
	byte vol_dev;
	byte dot_prob;
	byte target;
	byte on;
};

class avals
{
public:
};

#pragma pack()


////////////////////////////////////////////////////////////////////////
// Machine info

CMachineInfo const MacInfo =
{
	MT_GENERATOR,					//Machine type
		MI_VERSION,						//Interface version
		MIF_NO_OUTPUT | MIF_CONTROL_MACHINE, //Flags. MIF_NO_OUTPUT adds an ugly pan bar!
		1,								//Min tracks
		MAX_TRACKS,							//Max tracks
		NUMG,							//Number of global parameters
		NUMT,							//Number of track parameters
		pParameters,					//Pointer to parameters
		NUMA,								//Number of attributes
		pAttributes,					//Pointer to attributes
		FULL_NAME,			//Full name
		MACHINE_NAME,						//Short name
		AUTHOR,					//Author
		"/Assign Generator\n"
		"About " MACHINE_NAME	//Commands
};

inline int GetParamValue(CPeerCtrl *ct, float val);
inline int GetParamNoValue(CPeerCtrl *ct);
inline int GetParamMax(CPeerCtrl *ct);
inline int GetParamMin(CPeerCtrl *ct);



////////////////////////////////////////////////////////////////////////
// The "miex" class

class mi;

class miex : public CMDKMachineInterfaceEx {
public:
	mi *pmi; // pointer to 'parent'
	virtual void GetSubMenu(int const i, CMachineDataOutput *pout);
};


class CTrack {
public:
	int note;
	float volume;
	int prob;
	int centre_note, oct_dev;
	float vol_dev;
	float dot_prob;
	int target;
	int on;
	
	void Init() {
		on = false;
		note = NOTE_NO;
		volume = 80.0f / PARA_VOLUME_MAX;
		vol_dev = 0.0f;
		prob = 100;
		centre_note = 60;
	}
	void Stop() {
		on = false;
	}
};

////////////////////////////////////////////////////////////////////////

// probability functions

int weighted_bool(int n) {
	if (n > ((rand() % 100))) {
		return 1;
	} else {
		return 0;
	}
}

int weighted_bool(float n) {
	if (n > ((rand() % 100))) {
		return 1;
	} else {
		return 0;
	}
}
////////////////////////////////////////////////////////////////////////

// conversion functions

inline int oct_pitch_to_midi(int octave, int pitch) {
	return octave * 12 + pitch;
}


inline int midi_to_pitch(int midi_note) {
	return midi_note % 12;
}


inline int midi_to_oct(int midi_note) {
	return midi_note / 12;
}


inline int buzz_to_midi(int buzz_note) {
	if ((buzz_note % 16) > 11) {
		// it's not a real buzz note! hehe
		buzz_note += (16 - (buzz_note % 16)); // ie push it up to next C FIXME - not right!
	}
	return ((buzz_note >> 4) * 12 + (buzz_note & 0x0f) - 1);
}


inline int midi_to_buzz(int midi_note) {
	return (16 * ((midi_note - (midi_note % 12)) / 12) + midi_note % 12 + 1);
}

inline char * pitch_to_string(int pitch) {
	char *txt = "     ";
	char letter;
	int sharp = 1;
	switch (pitch) {
	case c:
		sharp = 0;
	case c_sh:
		letter = 'C'; break;
	case d:
		sharp = 0;
	case d_sh:
		letter = 'D'; break;
	case e:
		sharp = 0;
		letter = 'E'; break;
	case f:
		sharp = 0;
	case f_sh:
		letter = 'F'; break;
	case g:
		sharp = 0;
	case g_sh:
		letter = 'G'; break;
	case a:
		sharp = 0;
	case a_sh:
		letter = 'A'; break;
	case b:
		sharp = 0;
		letter = 'B'; break;
	case off:
		// has to be dealt with specially
		break;
	default:
		break;
	}
	if (!sharp) {
		sprintf(txt, "%c", letter);
	} else {
		sprintf(txt, "%c%c", letter, '#');
	}
	if (pitch == off) {
		sprintf(txt, "Off");
	}
	return txt;
}


inline char * oct_pitch_to_string(int octave, int pitch) {
	char *txt = "     ";
	sprintf(txt, "%s%d", pitch_to_string(pitch), octave);
	if (pitch == off) {
		sprintf(txt, "Off");
	}

	return txt;
}

/////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////
// The main machine interface class


class mi : public CMDKMachineInterface
{
public:
	mi();
	~mi();
	virtual void Tick();
	virtual void Stop();
	virtual void MDKInit(CMachineDataInput * const pi);
	virtual bool MDKWork(float *psamples, int numsamples, int const mode);
	bool MDKWorkStereo(float *psamples, int numsamples, int const mode) { return false; };
	virtual void MDKSave(CMachineDataOutput * const po);
	virtual char const *DescribeValue(int const param, int const value);
	CMDKMachineInterfaceEx *GetEx() { return &ex; }
	virtual void Command(const int i);
	void OutputModeChanged(bool stereo) {}
	virtual void SetNumTracks(const int n);
	virtual void mi::AttributesChanged();

	// i'm using the words pitch and note interchangeably here
	// bad style! maybe i'll clean this up sometime.
	int note_pick(int centre, int oct_dev) {
		int pitch, octave;
		float r = rand() / (float) RAND_MAX;
		float bin = 0.0f;
		int sigma = 0; // sum of relative probabilities
		
		for (int i = 0; i < numNotes; i++) {
			sigma += note_prob[i];
		}

		for (i = 0; i < numNotes; i++) {
			bin += note_prob[i] / (float) sigma;
			if (r < bin) {
				pitch = i;
				break;
			}
		}

		// note-off!
		if (pitch == off) return NOTE_OFF;

		int centre_note = midi_to_pitch(buzz_to_midi(centre));
		octave = midi_to_oct(buzz_to_midi(centre));

		// first, transform so we're as close to target note as possible.
		if (centre_note - pitch > 6) {
			octave++;
		} else if (pitch - centre_note > 6) {
			octave--;
		}

		// centred-mode:
		if (oct_dev <= 10) {
			r = (2 * rand() / (float) RAND_MAX) - 1.0f; // r is between -1 and 1
		} else if (oct_dev <= 20) { // plus-mode:
			oct_dev -= 10;
			r = (rand() / (float) RAND_MAX); // r is between 0 and 1
		} else { // minus-mode:
			oct_dev -= 20;
			r = - (rand() / (float) RAND_MAX); // r is between -1 and 0
		}

		octave += round(oct_dev * r * r * r);

		//printf("adding %d\n", round(oct_dev * r * r * r));
		if (octave > 9) octave = 9;
		if (octave < 0) octave = 0;

		// convert from octave-pitch to midi to buzz
		return midi_to_buzz(oct_pitch_to_midi(octave, pitch));
	}

	int round(float in) {
		//printf("round: in=%f; out=%d\n", in, (int) floor(in + 0.5f));
	    return (int) floor(in + 0.5f);
	}

	float vol_rand(float vol, float dev) {
		float r = (2.0f * rand() / (float) RAND_MAX) - 1.0f; // r is between -1 and 1
		vol += dev * r * r * r;
		if (vol > 1.0f) vol = 1.0f;
		if (vol < 0.0f) vol = 0.0f;
		return vol;
	}
	
	miex ex;
	CMachine *ThisMac;
	int numTracks;
	
	bool Initialising;
	
	gvals gval;
	tvals tval[MAX_TRACKS];
	avals aval;
	
	CPeerCtrl *note_ctrl[MAX_TRACKS];
	CPeerCtrl *vol_ctrl[MAX_TRACKS];
	CPeerCtrlNote *note_container[MAX_TRACKS];

	CTrack tracks[MAX_TRACKS];

	int note_prob[numNotes];
};



DLL_EXPORTS

////////////////////////////////////////////////////////////////////////
// [Con/de]structors
mi::mi() 
{
	GlobalVals = &gval;
	TrackVals = &tval;
	AttrVals = (int *)&aval;
	ex.pmi = this;
	numTracks = 0;
	srand(time(NULL));

	typedef char name_type[32];
	name_type *name = new name_type[numNotes];
	typedef char desc_type[64];
	desc_type *desc = new desc_type[numNotes];
	for (int i = 0; i < numNotes; i++) {
		sprintf(name[i], "Prob(%s)", pitch_to_string(i));
		sprintf(desc[i], "RELATIVE Probability note %s will be played", pitch_to_string(i));

		paraNoteProb[i].Type = pt_byte;

		paraNoteProb[i].Name = name[i];
		paraNoteProb[i].Description = desc[i];
		paraNoteProb[i].MinValue = 0;
		paraNoteProb[i].MaxValue = 100;
		paraNoteProb[i].NoValue = 101;
		paraNoteProb[i].Flags = MPF_STATE;
	}
	paraNoteProb[c].DefValue = 20;
	paraNoteProb[c_sh].DefValue = 0;
	paraNoteProb[d].DefValue = 25;
	paraNoteProb[d_sh].DefValue = 0;
	paraNoteProb[e].DefValue = 25;
	paraNoteProb[f].DefValue = 15;
	paraNoteProb[f_sh].DefValue = 0;
	paraNoteProb[g].DefValue = 10;
	paraNoteProb[g_sh].DefValue = 0;
	paraNoteProb[a].DefValue = 30;
	paraNoteProb[a_sh].DefValue = 0;
	paraNoteProb[b].DefValue = 10;
	paraNoteProb[off].DefValue = 10;
}

mi::~mi()
{
	int m;
	for (m = 0; m < MAX_TRACKS; m++) {
		if (note_ctrl[m]) delete note_ctrl[m];
		if (vol_ctrl[m]) delete vol_ctrl[m];
		if (note_container[m]) delete note_container[m];
	}
}

////////////////////////////////////////////////////////////////////////

void mi::MDKInit (CMachineDataInput * const pi)
{
	int m;
	Initialising = true;
	
	ThisMac = pCB->GetThisMachine();
	
	for (m = 0; m < MAX_TRACKS; m++) {
		tracks[m].Init();

		note_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) note_ctrl[m]->ReadFileData(pi);
		vol_ctrl[m] = new CPeerCtrl(ThisMac, this);
		if (pi) vol_ctrl[m]->ReadFileData(pi);

		note_container[m] = new CPeerCtrlNote(
			note_ctrl[m], vol_ctrl[m], NULL, NULL);
	}
	//AllocConsole();freopen("CON", "w", stdout);
	//FIXME: think of a way to use this!
	// could make a machine where user was allowed to type in commands or values
	// to a console and read results.

}

void mi::MDKSave(CMachineDataOutput * const po) {
	int m;
	for (m = 0; m < MAX_TRACKS; m++) {
		note_ctrl[m]->WriteFileData(po);
		vol_ctrl[m]->WriteFileData(po);
	}
}

void mi::SetNumTracks(int const n) {
	if (numTracks < n) {
		for (int i = numTracks; i < n; i++) {
			tracks[i].Init();
		}
	} else if (n < numTracks) {
		for (int i = n; i < numTracks; i++)
			tracks[i].Stop();
	}
	numTracks = n;
}

void mi::Stop() {
	for (int m = 0; m < MAX_TRACKS; m++) {
		tracks[m].Stop();
	}
}

void mi::AttributesChanged() {}

////////////////////////////////////////////////////////////////////////

inline int GetParamValue(CPeerCtrl *ct, float val)
{
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MinValue + (int)(val * (mpar->MaxValue - mpar->MinValue));
}

inline int GetParamNoValue(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->NoValue;
}

inline int GetParamMax(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MaxValue;
}

inline int GetParamMin(CPeerCtrl *ct) {
	const CMachineParameter *mpar = ct->GetParamInfo();
	return mpar->MinValue;
}

///////////////////////////////////////////////////////////////////////

void mi::Tick() {
	for (int i = 0; i < numNotes; i++) {
		if (gval.note_prob[i] != paraNoteProb[i].NoValue) {
			note_prob[i] = gval.note_prob[i];
		}
	}
	for (int t = 0; t < numTracks; t++) {
		if (tval[t].centre_note != paraCentre.NoValue) {
			tracks[t].centre_note = tval[t].centre_note;
		}
		if (tval[t].oct_dev != paraOctDev.NoValue) {
			tracks[t].oct_dev = tval[t].oct_dev;
		}
		if (tval[t].vol_dev != paraVolDev.NoValue) {
			tracks[t].vol_dev = tval[t].vol_dev / 100.0f;
		}
		if (tval[t].dot_prob != paraDotProb.NoValue) {
			// paraDotProb.Max is 254
			tracks[t].dot_prob = 100.0f 
				* tval[t].dot_prob * tval[t].dot_prob / (254.0f * 254.0f);
		}
		if (tval[t].target != paraTarget.NoValue) {
			tracks[t].target = tval[t].target;
		}
		if (tval[t].on != paraOn.NoValue) {
			tracks[t].on = tval[t].on;
		}
	}
	if (!Initialising) {
		// have to call CheckMachine() every tick for every CPeerCtrl object!
		// this is as good a place as any...
		for (int c = 0; c < MAX_TRACKS; c++) {
			note_ctrl[c]->CheckMachine();
			vol_ctrl[c]->CheckMachine();
		}
		for (int t = 0; t < numTracks; t++) {
			if (tracks[t].on) {

				// check note parameter and probability parameter
				int note, play_note;
				
				if (tval[t].note != paraNote.NoValue) {
					note = tval[t].note;
				} else {
					note = note_pick(tracks[t].centre_note, tracks[t].oct_dev);
				}
				
				
				if (tval[t].prob == paraProb.NoValue) {
					if (tval[t].note != paraNote.NoValue) { // definitely play
						play_note = 1;
					} else { // play with probability dot_prob
						play_note = weighted_bool(tracks[t].dot_prob);
					}
				} else { // play with probability prob
					play_note = weighted_bool(tval[t].prob);
				}
				
				if (play_note) {
					if (note == NOTE_OFF) { // best not to send volume with it.
						if (note_ctrl[t]->GotParam()) {
							note_ctrl[t]->ControlChange_Immediate(tracks[t].target, NOTE_OFF);
						}
					} else {
						if (tval[t].volume != paraVolume.NoValue) {
							tracks[t].volume = (float) tval[t].volume / PARA_VOLUME_MAX;
						}
						
						int vol_to_send;
						if (vol_ctrl[t]->GotParam()) {
							float float_vol = vol_rand(tracks[t].volume, tracks[t].vol_dev);
							vol_to_send = GetParamValue(vol_ctrl[t], 
								vol_rand(tracks[t].volume, tracks[t].vol_dev));
						} else { // vol_ctrl[t] isn't assigned, so
							vol_to_send = 0; // doesn't matter what we send
						}
						
						//printf("sending note: %d, volume: %d to track: %d\n",
						//	note, vol_to_send, tracks[t].target);
						if (note_ctrl[t]->GotParam()) {
							note_container[t]->ControlChange_Immediate(tracks[t].target,
								note, vol_to_send, 0, 0);
						}
					}
				}
			}
		}
	}	
}	


////////////////////////////////////////////////////////////////////////

bool mi::MDKWork(float *psamples, int numsamples, const int mode) {
	
	// do nothing! nice.
	
	Initialising = false;
	return false;
}

////////////////////////////////////////////////////////////////////////

HINSTANCE dllInstance;
mi *g_mi;
CPeerCtrl *g_note_ctrl, *g_vol_ctrl;

BOOL WINAPI DllMain ( HANDLE hModule, DWORD fwdreason, LPVOID lpReserved )
{
	switch (fwdreason) {
	case DLL_PROCESS_ATTACH:
		dllInstance = (HINSTANCE) hModule;
		break;
		
	case DLL_THREAD_ATTACH: break;
	case DLL_THREAD_DETACH: break;
	case DLL_PROCESS_DETACH: break;
	}
	return TRUE;
}


////////////////////////////////////////////////////////////////////////
// layer dialog code

bool DoEndAssignDlg (HWND hDlg, bool ShouldSave) {
	//Store the parameters chosen (if any) in the dialog, or not
	//Return value: true if should close dialog, false if not
	
	if (ShouldSave) {
		char MacName[128];
		char NoteName[255];
		char VolName[255];

		if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0) {
			// note
			int s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST),
					LB_GETTEXT, s, long(&NoteName));
				
				g_note_ctrl->AssignParameter(MacName, NoteName);
			} else {
				g_note_ctrl->UnassignParameter();
			}
			// volume
			s = SendMessage(GetDlgItem(hDlg,IDC_PARALIST2), 
				LB_GETCURSEL, 0, 0);
			if (s != LB_ERR) {
				SendMessage(GetDlgItem(hDlg,IDC_PARALIST2),
					LB_GETTEXT, s, long(&VolName));
				
				g_vol_ctrl->AssignParameter(MacName, VolName);
			} else {
				g_vol_ctrl->UnassignParameter();
			}
		} else {
			g_note_ctrl->UnassignParameter();
			g_vol_ctrl->UnassignParameter();
		}
		
		return true;
	} else {
		return true;
	}
}

BOOL APIENTRY AssignDialog(HWND hDlg, UINT uMsg, WPARAM wParam, LPARAM lParam) 
//The big procedure for the GUI dialog
{
	int m = 0;
	switch(uMsg) {
	case WM_INITDIALOG:
		char txt[128];
		char mac_name[128];
		char note_name[128];
		char vol_name[128];

		sprintf(note_name, g_note_ctrl->GetAssignmentString());
		sprintf(mac_name, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		sprintf(vol_name, g_vol_ctrl->GetAssignmentString());
		sprintf(txt, "%s: note: %s; volume: %s",
			mac_name, note_name, vol_name);

		SetDlgItemText(hDlg, IDC_THISMACINFO, txt);
		
		//Populate the boxes - only need to do this once so use g_note_ctrl for it.
		g_note_ctrl->GetMachineNamesToCombo(hDlg, 
			IDC_MACHINECOMBO, g_mi->pCB->GetMachineName(g_mi->ThisMac));
		
		// note
		if (g_note_ctrl->GotParam())
		{
			//select machine and parameter
			if (g_note_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_note_ctrl->GetParamNamesToList(
					g_note_ctrl->GetMachine(),
					hDlg, IDC_PARALIST, ALLOW_NOTE | ALLOW_ALL_GROUPS);
				
				g_note_ctrl->SelectParameterInList(hDlg, IDC_PARALIST);
				
			}
		}
			
		// volume
		if (g_vol_ctrl->GotParam()) 
		{
			//select machine and parameter
			if (g_vol_ctrl->SelectMachineInCombo(hDlg, IDC_MACHINECOMBO))
			{
				g_vol_ctrl->GetParamNamesToList(
					g_vol_ctrl->GetMachine(),
					hDlg, IDC_PARALIST2, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
				
				g_vol_ctrl->SelectParameterInList(hDlg, IDC_PARALIST2);
			}
		}
		
		return 1;
	case WM_SHOWWINDOW: 
		return 1;
		
	case WM_CLOSE:
		if (DoEndAssignDlg(hDlg, false))
			EndDialog (hDlg, TRUE);
		return 0;
		
	case WM_COMMAND:
		//If a button or other control is clicked or something
		switch (LOWORD (wParam))
		{
		case IDOK:
			if (DoEndAssignDlg(hDlg, true))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDCANCEL:
			if (DoEndAssignDlg(hDlg, false))
				EndDialog(hDlg, TRUE);
			return 1;
			
		case IDC_DEASSIGN:
			g_note_ctrl->UnassignParameter();
			g_vol_ctrl->UnassignParameter();
			return 1;
			
		case IDC_MACHINECOMBO:
			if (HIWORD(wParam) == CBN_SELCHANGE) { //selection is changed
				char MacName[128];
				
				if (GetDlgItemText(hDlg, IDC_MACHINECOMBO, MacName, 128) != 0)
				{	//ie if a machine is selected
					//Populate parameter list
					g_note_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST, ALLOW_NOTE | ALLOW_ALL_GROUPS);
					g_vol_ctrl->GetParamNamesToList(
						g_mi->pCB->GetMachine(MacName),
						hDlg, IDC_PARALIST2, ALLOW_ALL_TYPES | ALLOW_ALL_GROUPS);
					
				}
				else {
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST), LB_RESETCONTENT, 0,0);
					SendMessage(GetDlgItem(hDlg, IDC_PARALIST2), LB_RESETCONTENT, 0,0);
				}
				
			}
			
			return 1;
			
		default:
			return 0;
		}
		break;
	}
	return 0;
}

void miex::GetSubMenu(int const i, CMachineDataOutput *pout) {
	char txt[256];
	int m = 0;
	switch (i) {
	case 0:
		// assign submenu

		// i declare space for the _name strings because
		// in case 0 here, we would get eg "Primifun->Note; Primifun->Note" if we
		// called GetAssignmentString() directly in the sprintf(txt) call. i think.
		char note_name[128];
		char vol_name[128];
		for (m = 0; m < pmi->numTracks; m++) {
			sprintf(note_name, pmi->note_ctrl[m]->GetAssignmentString());
			sprintf(vol_name, pmi->vol_ctrl[m]->GetAssignmentString());
			sprintf(txt, "Generator %i: %s; %s",
				m, note_name, vol_name);
			pout->Write(txt);
		}		
		break;
	case 1:
		// about - no submenu
		break;
	default:
		break;
	}
}	

void mi::Command(const int i) {	
	if (i == 0) { // can't happen - 0 is a submenu
	} else if (i == 1) {
		MessageBox(NULL, // dunno what NULL signifies
			ABOUT_TEXT, // message text
			"About " MACHINE_NAME, // window title
			MB_OK|MB_SYSTEMMODAL); // an ok button
	} else if (i >= 256 && i < 256 * 2) { // layers
		int j = i - 256;
		assert(j >= 0 && j <= MAX_TRACKS);
		g_mi = this;
		g_note_ctrl = note_ctrl[j];
		g_vol_ctrl = vol_ctrl[j];
		DialogBox(dllInstance, MAKEINTRESOURCE (IDD_ASSIGN_NOTE), GetForegroundWindow(), (DLGPROC) &AssignDialog);
		note_container[j]->note_ctrl = note_ctrl[j];
		note_container[j]->vol_ctrl = vol_ctrl[j];
	}
}

////////////////////////////////////////////////////////////////////////

char const *mi::DescribeValue(const int param, const int value)
{
	static char txt[16];
	
	switch (param) {
	case numNote: // note
		sprintf(txt, "%s", oct_pitch_to_string(midi_to_oct(buzz_to_midi(value)),
			midi_to_pitch(buzz_to_midi(value))));
		return txt;
	case numVolume: // volume
		sprintf(txt, "%i%%", (int) (value * 100.0f / PARA_VOLUME_MAX));
		return txt;
	case numProb: // prob
		sprintf(txt, "%i", value);
		return txt;
	case numCentre:
		sprintf(txt, "%s", oct_pitch_to_string(midi_to_oct(buzz_to_midi(value)),
			midi_to_pitch(buzz_to_midi(value))));
		return txt;
	case numVolDev:
		sprintf(txt, "%d%%", value);
		return txt;
	case numOctDev: // lala
		if (value <= 10) {
			sprintf(txt, "+/- %i", value);
		} else if (value <= 20) {
			sprintf(txt, "+ %i", value - 10);
		} else {
			sprintf(txt, "- %i", value - 20);
		}
		return txt;
	case numDotProb: // 
		sprintf(txt, "%.2f%%", 100.0f * value * value / (254.0f * 254.0f));
		return txt;
	case numTarget: // target track
		sprintf(txt, "%d", value);
		return txt;
	case numOn:
		if (value) {
			sprintf(txt, "On");
		} else {
			sprintf(txt, "Off");
		}
		return txt;
	default: // includes note-probabilities
		sprintf(txt, "%d%%", value);
		return txt;
	}
}

